#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>

#include "addrbook.h"
#include "general.h"
#include "scrolbox.h"
#include "scrsaver.h"
#include "screen.h"
#include "keyb.h"
#include "files.h"
#include "vars.h"

ADDR_REC tempaddr;

static ADDR_REC *first_addr;
static ADDR_REC *addr;

static MAIL_REC *first_mail;
static MAIL_REC *mail_addr;

static unsigned address, mail_address;

void put_field(char *str, char *output, int size)
{
    str = strchr(str,'=');
    if (str == NULL) return;

    str++;
    while (*str != '\0' && (*str == ' ' || *str == 9)) str++;
    strncpy(output,str,size-1);
    output[size-1] = '\0';
}

void put_short(char *str, unsigned short *output)
{
    str = strchr(str,'=');
    if (str == NULL) return;

    str++;
    while (*str != '\0' && (*str == ' ' || *str == 9)) str++;
    sscanf(str,"%hu",output);
}

void put_bool(char *str, char *output)
{
    str = strchr(str,'=');
    if (str == NULL) return;

    str++;
    while (*str != '\0' && (*str == ' ' || *str == 9)) str++;
    *str = toupper(*str);
    *output = (*str == 'Y' || *str == 'T' || *str == '1');
}

void put_maillist(char *str)
{
    MAIL_LINK_REC *tmplist,*prevlist;
    ADDR_REC *addr;
    unsigned pos;

    str = strchr(str,'=');
    if (str == NULL) return;

    str++;
    while (*str != '\0' && (*str == ' ' || *str == 9)) str++;

    tmplist = mail_addr->maillist; prevlist = NULL;
    while (tmplist != NULL)
    {
        prevlist = tmplist;
        tmplist = tmplist->next;
    }

    tmplist = (MAIL_LINK_REC *) malloc(sizeof(MAIL_LINK_REC));
    tmplist->next = NULL;

    addr = first_addr; pos = 1;
    while (addr != NULL)
    {
        if (stricmp(addr->quickkey,str) == 0)
        {
            tmplist->addrnum = pos;
            break;
        }
        addr = addr->next; pos++;
    }

    if (addr != NULL)
    {
        if (mail_addr->maillist == NULL) mail_addr->maillist = tmplist;
        if (prevlist != NULL) prevlist->next = tmplist;
    }
    else
    {
        free(tmplist);
    }
}

/* Read address book to memory */
void read_addrbook(unsigned *addresses, unsigned *mail_addresses)
{
    FILE *Faddr;
    ADDR_REC *tmprec;
    MAIL_REC *tmpmail;
    char tmp[256],waiting;

    if (addresses != NULL) *addresses = 0;
    if (mail_addresses != NULL) *mail_addresses = 0;
    Faddr = fopen_ign(strcat(strcpy(tmp,setup.datapath),"addrbook.dat"),"r+t");
    if (Faddr == NULL) return;

    first_addr = NULL; addr = NULL;
    first_mail = NULL; mail_addr = NULL;

    waiting = 1;
    while (_fgets(tmp,sizeof(tmp),Faddr) != NULL)
    {
        if (tmp[0] == ';') continue; /* Comment */
        if (stricmp(tmp,"[address]") == 0)
        {
            /* New address book record */
            tmprec = (ADDR_REC *) malloc(sizeof(ADDR_REC));
            if (tmprec == NULL)
            {
                err_box("Not enough memory!");
                return;
            }
            memset(tmprec,0,sizeof(ADDR_REC));

            if (first_addr == NULL) first_addr = tmprec;
            if (addr != NULL) addr->next = tmprec;
            tmprec->prev = addr;
            tmprec->next = NULL;
            addr = tmprec;

            waiting = 0;
            if (addresses != NULL) (*addresses)++;
            continue;
        }

        if (stricmp(tmp,"[mailing_list]") == 0)
        {
            /* New address book record */
            tmpmail = (MAIL_REC *) malloc(sizeof(MAIL_REC));
            if (tmpmail == NULL)
            {
                err_box("Not enough memory!");
                return;
            }
            memset(tmpmail,0,sizeof(MAIL_REC));

            if (first_mail == NULL) first_mail = tmpmail;
            if (mail_addr != NULL) mail_addr->next = tmpmail;
            tmpmail->prev = mail_addr;
            tmpmail->next = NULL;
            mail_addr = tmpmail;

            waiting = 2;
            if (mail_addresses != NULL) (*mail_addresses)++;
            continue;
        }

        if (waiting == 1) continue;

        if (waiting == 0)
        {
            if (strnicmp(tmp,"QuickKey ",9) == 0) put_field(tmp,addr->quickkey,sizeof(addr->quickkey));
            if (strnicmp(tmp,"Comment ",8) == 0) put_field(tmp,addr->comment,sizeof(addr->comment));
            if (strnicmp(tmp,"To ",3) == 0) put_field(tmp,addr->to_field,sizeof(addr->to_field));
            if (strnicmp(tmp,"Zone ",5) == 0) put_short(tmp,&addr->zone);
            if (strnicmp(tmp,"Net ",4) == 0) put_short(tmp,&addr->net);
            if (strnicmp(tmp,"Node ",5) == 0) put_short(tmp,&addr->node);
            if (strnicmp(tmp,"Point ",6) == 0) put_short(tmp,&addr->point);
            if (strnicmp(tmp,"INet ",4) == 0) put_field(tmp,addr->inet,sizeof(addr->inet));
            if (strnicmp(tmp,"Subject ",8) == 0) put_field(tmp,addr->subject,sizeof(addr->subject));
            if (strnicmp(tmp,"FirstLine ",10) == 0) put_field(tmp,addr->first_line,sizeof(addr->first_line));
        }
        else
        {
            if (strnicmp(tmp,"QuickKey ",9) == 0) put_field(tmp,mail_addr->quickkey,sizeof(mail_addr->quickkey));
            if (strnicmp(tmp,"Comment ",8) == 0) put_field(tmp,mail_addr->comment,sizeof(mail_addr->comment));
            if (strnicmp(tmp,"Subject ",8) == 0) put_field(tmp,mail_addr->subject,sizeof(mail_addr->subject));
            if (strnicmp(tmp,"BlindCarbon ",12) == 0) put_bool(tmp,&mail_addr->blind);
            if (strnicmp(tmp,"MailList ",9) == 0) put_maillist(tmp);
        }
    }

    fclose(Faddr);

    addr = first_addr;
    mail_addr = first_mail;
    address = 1;
    mail_address = 1;
}

/* Save address book to disk */
void save_addrbook(void)
{
    FILE *Faddr;
    char tmp[256];
    MAIL_LINK_REC *tmplist;

    Faddr = FileOpen(strcat(strcpy(tmp,setup.datapath),"addrbook.dat"),"w+t");
    if (Faddr == NULL) return;

    addr = first_addr;
    while (addr != NULL)
    {
        fprintf(Faddr,"[address]\n\n");
        if (addr->quickkey[0] != '\0') fprintf(Faddr,"QuickKey = %s\n",addr->quickkey);
        if (addr->comment[0] != '\0') fprintf(Faddr,"Comment = %s\n",addr->comment);
        if (addr->to_field[0] != '\0') fprintf(Faddr,"To = %s\n",addr->to_field);
        if (addr->zone != 0) fprintf(Faddr,"Zone = %u\n",addr->zone);
        if (addr->net != 0) fprintf(Faddr,"Net = %u\n",addr->net);
        if (addr->node != 0) fprintf(Faddr,"Node = %u\n",addr->node);
        if (addr->point != 0) fprintf(Faddr,"Point = %u\n",addr->point);
        if (addr->inet[0] != '\0') fprintf(Faddr,"INet = %s\n",addr->inet);
        if (addr->subject[0] != '\0') fprintf(Faddr,"Subject = %s\n",addr->subject);
        if (addr->first_line[0] != '\0') fprintf(Faddr,"FirstLine = %s\n",addr->first_line);
        fprintf(Faddr,"\n");
        addr = addr->next;
    }
    address = 1; addr = first_addr;

    mail_addr = first_mail;
    while (mail_addr != NULL)
    {
        fprintf(Faddr,"[mailing_list]\n\n");
        if (mail_addr->quickkey[0] != '\0') fprintf(Faddr,"QuickKey = %s\n",mail_addr->quickkey);
        if (mail_addr->comment[0] != '\0') fprintf(Faddr,"Comment = %s\n",mail_addr->comment);
        if (mail_addr->subject[0] != '\0') fprintf(Faddr,"Subject = %s\n",mail_addr->subject);
        if (mail_addr->blind) fprintf(Faddr,"BlindCarbon = Yes\n");
        tmplist = mail_addr->maillist;
        while (tmplist != NULL)
        {
            if (address > tmplist->addrnum)
            {
                addr = first_addr;
                address = 1;
            }
            while (addr != NULL && address < tmplist->addrnum)
            {
                addr = addr->next;
                address++;
            }
            fprintf(Faddr,"MailList = %s\n",addr->quickkey);
            tmplist = tmplist->next;
        }
    }

    fclose(Faddr);

    addr = first_addr;
    address = 1;
}

/* Release memory used by address book */
void close_addrbook(void)
{
    MAIL_LINK_REC *tmplist;

    while (first_addr != NULL)
    {
        addr = first_addr->next;
        free(first_addr);
        first_addr = addr;
    }

    while (first_mail != NULL)
    {
        mail_addr = first_mail->next;
        while (first_mail->maillist != NULL)
        {
            tmplist = first_mail->maillist;
            free(first_mail->maillist);
            first_mail->maillist = tmplist;
        }
        free(first_mail);
        first_mail = mail_addr;
    }
}

/* Adopt new info to address book */
int adopt_address(char place, int email)
{
    unsigned short cx,cy;
    char *oldscr;

    int oldkeys[sizeof(keys)],oldkeypos[sizeof(keypos)],oldkeyb_keys;
    int info,ypos,size,quit;
    char *pstr,str[256],*pstr2,ch;

    ADDR_REC addr_rec,*tmp_addr;

    char *pointers[] =
    {
        addr_rec.quickkey,
        addr_rec.comment,
        addr_rec.to_field,
        NULL,
        addr_rec.inet,
        addr_rec.subject,
        addr_rec.first_line
    };

    char sizes[] =
    {
        sizeof(addr_rec.quickkey)-1,
        sizeof(addr_rec.comment)-1,
        sizeof(addr_rec.to_field)-1,
        0,
        sizeof(addr_rec.inet)-1,
        sizeof(addr_rec.subject)-1,
        sizeof(addr_rec.first_line)-1
    };

    if (place == 0)
    {
        /* Adopt new address */
        read_addrbook(NULL,NULL);

        memset(&addr_rec,0,sizeof(addr_rec));

        if (fmt != format_soup)
            strcpy(addr_rec.to_field,msg.mfrom);
        else
        {
            pstr = strchr(msg.mfrom, '<');
            if (pstr != NULL)
            {
                /* name <inet@address> */
                *pstr = '\0';
                strcpy(addr_rec.to_field,msg.mfrom);
                *pstr++ = '<';
                pstr2 = strchr(msg.mfrom, '>');
                if (pstr2 != NULL) *pstr2 = '\0';
                strcpy(addr_rec.inet, pstr);
                if (pstr2 != NULL) *pstr2 = '>';
            }
            else
            {
                pstr = strchr(msg.mfrom, '(');
                if (pstr != NULL)
                {
                    /* inet@address (name) */
                    pstr2 = pstr++;
                    while (*(pstr2-1) == ' ') pstr2--; ch = *pstr2; *pstr2 = '\0';
                    strcpy(addr_rec.inet, msg.mfrom); *pstr2 = ch;
                    pstr2 = strchr(pstr, ')');
                    if (pstr2 != NULL) *pstr2 = '\0';
                    strcpy(addr_rec.to_field, pstr);
                    if (pstr2 != NULL) *pstr2 = ')';
                }
                else
                {
                    /* inet@address */
                    strcpy(addr_rec.inet, msg.mfrom);
                }
            }
        }
        addr_rec.zone = msg.zone;
        addr_rec.net = msg.net;
        addr_rec.node = msg.node;
        addr_rec.point = pkt->fmpt;

        /* Get quickkey */
        pstr = strrchr(msg.mfrom,' ');
        if (pstr == NULL)
        {
            /* One word name */
            addr_rec.quickkey[0] = toupper(msg.mfrom[0]);
            addr_rec.quickkey[1] = '\0';
        }
        else
        {
            addr_rec.quickkey[0] = toupper(msg.mfrom[0]);
            addr_rec.quickkey[1] = toupper(*(pstr+1));
            addr_rec.quickkey[2] = '\0';
        }
    }
    else if (place == 1)
    {
        /* Modify existing address */
        memcpy(&addr_rec,addr,sizeof(addr_rec));
    }
    else if (place == 2)
    {
        /* Create new */
        memset(&addr_rec,0,sizeof(addr_rec));
    }
    else if (place == 3)
    {
        /* Adopt new address from reply */
        read_addrbook(NULL,NULL);

        memset(&addr_rec,0,sizeof(addr_rec));

        if (fmt != format_soup)
            strcpy(addr_rec.to_field,reply.mto);
        else if (email)
            strcpy(addr_rec.inet,reply.mto);

        addr_rec.zone = reply.destzone;
        addr_rec.net = reply.destnet;
        addr_rec.node = reply.destnode;
        addr_rec.point = reply.destpoint;

        /* Get quickkey */
        pstr = strrchr(reply.mto,' ');
        if (pstr == NULL)
        {
            /* One word name */
            addr_rec.quickkey[0] = toupper(reply.mto[0]);
            addr_rec.quickkey[1] = '\0';
        }
        else
        {
            addr_rec.quickkey[0] = toupper(reply.mto[0]);
            addr_rec.quickkey[1] = toupper(*(pstr+1));
            addr_rec.quickkey[2] = '\0';
        }
    }

    /* Save old keyb_keys, keys[] and keypos[] variables */
    memcpy(oldkeys,keys,sizeof(oldkeys));
    memcpy(oldkeypos,keypos,sizeof(oldkeypos));
    oldkeyb_keys = keyb_keys;

    save_scr(&cx,&cy,&oldscr);
    draw_shaded_box(scrwidth/2-39,8,scrwidth/2+40,21,color[col_info_frame],color[col_info_title],"Edit address book entry");
    cwritexy(scrwidth/2-37,10,lang[282],color[col_info_text]);
    cwritexy(scrwidth/2-37,11,lang[283],color[col_info_text]);
    cwritexy(scrwidth/2-37,12,lang[284],color[col_info_text]);
    cwritexy(scrwidth/2-37,13,lang[285],color[col_info_text]);
    cwritexy(scrwidth/2-37,14,lang[286],color[col_info_text]);
    cwritexy(scrwidth/2-37,15,lang[287],color[col_info_text]);
    cwritexy(scrwidth/2-37,17,lang[288],color[col_info_text]);
    sprintf(str," %s ",lang[289]);
    cmiddle(21,str,color[col_info_hilight]);
    keyb_keys = 2; keys[0] = 'S'; keys[1] = 'L';

    for (ypos=1; ypos<6; ypos++)
    {
        if (ypos != 3)
        {
            strcpy(str,pointers[ypos]);
            str[66] = 0;
        }
        else
        {
            sprintf(str,"%u:%u/%u.%u",addr_rec.zone,addr_rec.net,addr_rec.node,addr_rec.point);
        }
        cwritexy(scrwidth/2-27,10+ypos,str,color[col_info_hilight]);
    }

    strcpy(str,pointers[6]); str[76] = 0;
    cwritexy(scrwidth/2-37,18,str,color[col_info_hilight]);

    ypos = 0;
    quit = 0;
    while (!quit)
    {
        if (sizes[ypos] < 66) size = sizes[ypos]; else size = 66;
        pstr = pointers[ypos];
        switch (ypos)
        {
            case 3:
                sprintf(str,"%u:%u/%u.%u",addr_rec.zone,addr_rec.net,addr_rec.node,addr_rec.point);
                info = get_string(scrwidth/2-27,13,23,23,str,128+8+4,color[col_info_text],color[col_info_hilight],0);
                str2addr(str,&addr_rec.zone,&addr_rec.net,&addr_rec.node,&addr_rec.point);
                sprintf(str,"%u:%u/%u.%u",addr_rec.zone,addr_rec.net,addr_rec.node,addr_rec.point);
                tbar(scrwidth/2-17,13,scrwidth/2+6,13,color[col_info_hilight]);
                cwritexy(scrwidth/2-27,13,str,color[col_info_hilight]);
                break;
            case 6:
                info = get_string(scrwidth/2-37,18,sizes[ypos],76,pstr,128+8+4,color[col_info_text],color[col_info_hilight],0);
                break;
            default:
                info = get_string(scrwidth/2-27,10+ypos,sizes[ypos],size,pstr,128+8+4,color[col_info_text],color[col_info_hilight],0);
                break;
        }

        if (key_found != 0)
        {
            if (key_found == keys[0])
            {
                /* Alt-S - Get subject */
                if (place == 3)
                    strcpy(addr_rec.subject,reply.subj);
                else if (place == 0)
                    strcpy(addr_rec.subject,msg.subj);
                cwritexy(scrwidth/2-27,15,addr_rec.subject,color[col_info_text]);
            }
            else if (key_found == keys[1])
            {
                /* Alt-L - Get first line */
                if (place == 3 && pkt->first_linerec != NULL)
                    pkt->read_replyline(pkt, addr_rec.first_line, 1);
                else if (place == 0 && pkt->first_linerec != NULL)
                    pkt->read_line(pkt, addr_rec.first_line, 1);
                cwritexy(scrwidth/2-37,18,addr_rec.first_line,color[col_info_text]);
            }
            continue;
        }

        switch (info)
        {
            case 0:
            case 1:
                if (ypos < 6) ypos++;
                break;
            case 2:
                if (ypos > 0) ypos--;
                break;
            case 3:
                if (place != 1)
                {
                    addr = (ADDR_REC *) malloc(sizeof(ADDR_REC));
                    if (addr == NULL)
                    {
                        err_box("Not enough memory!");
                        quit = 1;
                        break;
                    }

                    tmp_addr = first_addr;
                    if (tmp_addr != NULL)
                    {
                        /* Seek to last record */
                        while (tmp_addr->next != NULL)
                            tmp_addr = tmp_addr->next;
                        tmp_addr->next = addr;
                    }
                    addr_rec.prev = tmp_addr;
                    addr_rec.next = NULL;
                    if (first_addr == NULL) first_addr = addr;
                }

                memcpy(addr,&addr_rec,sizeof(addr_rec));
                save_addrbook();
                quit = 2;
                break;
            case 5:
                quit = 1;
                break;
        }
    }

    if (place == 0) close_addrbook();
    old_scr(cx,cy,&oldscr);

    /* Restore keyb_keys, keys[] and keypos[] variables */
    memcpy(keys,oldkeys,sizeof(oldkeys));
    memcpy(keypos,oldkeypos,sizeof(oldkeypos));
    keyb_keys = oldkeyb_keys;

    return quit == 2;
}

/* Draw address book entry */
void draw_addr(unsigned ypos, unsigned pos)
{
    char ch;

    if (addr == NULL)
    {
        addr = first_addr;
        address = 1;
    }
    while (address < pos && addr != NULL)
    {
        addr = addr->next;
        address++;
    }

    while (address > pos && addr != NULL)
    {
        addr = addr->prev;
        address--;
    }

    tbar(scrwidth/2-37,ypos,scrwidth/2+37,ypos,color[col_box_text]);
    if (addr == NULL) return;

    cwritexy(scrwidth/2-37,ypos,addr->quickkey,color[col_box_text]);
    cwritexy(scrwidth/2-20,ypos,addr->to_field,color[col_box_hilight]);
    ch = addr->comment[31]; addr->comment[31] = '\0';
    cwritexy(scrwidth/2+5,ypos,addr->comment,color[col_box_text]);
    addr->comment[31] = ch;
}

/* Remove address book entry */
int delete_entry(SCROLL_BOX *scrbox)
{
    char *oldscr;
    unsigned short cx,cy;

    char ch;
    int killed,ypos,len;

    if (*scrbox->lines == 0) return 0;

    save_scr(&cx,&cy,&oldscr);
    killed = 0;
    len = get_max_langtext_len(277,279);
    draw_shaded_box(scrwidth/2-len/2-2,9,scrwidth/2+len/2+2,12,color[col_warn_frame],color[col_warn_title],lang[277]);
    cmiddle(10,lang[278],color[col_warn_text]);
    cmiddle(11,lang[279],color[col_warn_text]);
    tattrbar(scrwidth/2-len/2-1,10,scrwidth/2+len/2+1,10,color[col_warn_selectbar]);
    ypos = 0;
    for (;;)
    {
        if (sk_kbhit())
        {
            ch = sk_getch();
            if (ch == 13) break;
            if (ch == 27) { ypos = 1; break; }
            if (ch == 0)
            {
                ch = sk_getch();
                if (ch == 'H' || ch == 'P')
                {
                    tattrbar(scrwidth/2-len/2-1,10+ypos,scrwidth/2+len/2+1,10+ypos,color[col_warn_text]);
                    ypos = 1-ypos;
                    tattrbar(scrwidth/2-len/2-1,10+ypos,scrwidth/2+len/2+1,10+ypos,color[col_warn_selectbar]);
                }
            }
        }
        else
        {
            give_timeslice();
        }
    }

    if (ypos == 0)
    {
        /* Delete */
        killed = 1;
        if (addr->prev != NULL) addr->prev->next = addr->next;
        if (addr->next != NULL) addr->next->prev = addr->prev;
        if (first_addr == addr) first_addr = addr->next;
        free(addr);
        addr = first_addr; address = 1;
        (*scrbox->lines)--;
        if (scrbox->ypos+scrbox->upy > *scrbox->lines)
        {
            if (scrbox->ypos+scrbox->upy > 1)
            {
                if (scrbox->ypos > 1) scrbox->ypos--; else scrbox->upy--;
            }
        }
    }
    old_scr(cx,cy,&oldscr);
    return killed;
}

/* Insert new address book entry */
int insert_entry(SCROLL_BOX *scrbox)
{
    if (adopt_address(2,0))
    {
        (*scrbox->lines)++;
        return 1;
    }

    return 0;
}

/* Edit address book entry */
int edit_address_book(SCROLL_BOX *scrbox)
{
    if (scrbox->ypos+scrbox->upy > *scrbox->lines) return 0;
    return adopt_address(1,0);
}

unsigned addresses,mail_addresses;
unsigned addr_ypos,addr_upy;
unsigned mail_ypos,mail_upy;

int switch_box(SCROLL_BOX *scrbox)
{
    if (scrbox->ystart == 3)
    {
        addr_ypos = scrbox->ypos; addr_upy = scrbox->upy;
        scrbox->ypos = mail_ypos; scrbox->upy = mail_upy;
        scrbox->ystart = scrsize/2+2; scrbox->yend = scrsize/2-2;
        scrbox->lines = &mail_addresses;
//        scrbox->draw_func = (DRAW_FUNC) mail_draw_addr;
    }
    else
    {
        mail_ypos = scrbox->ypos; mail_upy = scrbox->upy;
        scrbox->ypos = addr_ypos; scrbox->upy = addr_upy;
        scrbox->ystart = 3; scrbox->yend = scrsize/2-2;
        scrbox->lines = &addresses;
        scrbox->draw_func = (DRAW_FUNC) draw_addr;
    }
    return 1;
}

/* Show address book */
int address_book(ADDR_REC *addr_out)
{
    unsigned short cx,cy;
    char *oldscr;

    SCROLL_BOX scrbox;
    unsigned pos;
    int ret;

    read_addrbook(&addresses,&mail_addresses);

    save_scr(&cx,&cy,&oldscr);
    draw_shaded_box(scrwidth/2-38,2,scrwidth/2+38,scrsize/2-1,color[col_box_frame],color[col_box_title],"Address book");
    draw_shaded_box(scrwidth/2-38,scrsize/2+1,scrwidth/2+38,scrsize-1,color[col_box_frame],color[col_box_title],"Address book");

    ret = 0;
    memset(&scrbox,0,sizeof(scrbox));
    mail_ypos = 1; mail_upy = 0;
    scrbox.ypos = 1; scrbox.upy = 0;
    scrbox.xstart = scrwidth/2-37; scrbox.xend = scrwidth/2+37;
    scrbox.ystart = 3; scrbox.yend = scrsize/2-2;
    scrbox.lines = &addresses;
    scrbox.scrollbar_color = color[col_box_select];
    scrbox.background_color = color[col_box_text];
    scrbox.draw_func = (DRAW_FUNC) draw_addr;
    add_key(0, 'S', (SCROLL_FUNC) delete_entry, &scrbox);
    add_key(0, 'R', (SCROLL_FUNC) insert_entry, &scrbox);
    add_key(0, 18, (SCROLL_FUNC) edit_address_book, &scrbox);
    add_key(9, 0, (SCROLL_FUNC) switch_box, &scrbox);
    while ((pos = scroll_box(&scrbox)) > 0)
    {
        /* Edit address entry */
        if (addr == NULL)
        {
            addr = first_addr;
            address = 1;
        }
        while (address < pos && addr != NULL)
        {
            addr = addr->next;
            address++;
        }

        while (address > pos && addr != NULL)
        {
            addr = addr->prev;
            address--;
        }

        if (addr == NULL) continue;

        if (addr_out != NULL)
        {
            memcpy(addr_out,addr,sizeof(ADDR_REC));
            ret = 1;
            break;
        }
        adopt_address(1,0);
    }
    deinit_scroll_box(&scrbox);

    old_scr(cx,cy,&oldscr);

    save_addrbook();
    close_addrbook();

    return ret;
}

int get_quickkey(char maillist, char *key)
{
    ADDR_REC *addr;

    read_addrbook(NULL,NULL);

    addr = first_addr;
    while (addr != NULL)
    {
        if (stricmp(key,addr->quickkey) == 0)
        {
            /* Found it! */
            memcpy(&tempaddr,addr,sizeof(tempaddr));
            break;
        }
        addr = addr->next;
    }

    close_addrbook();

    return addr != NULL;
}
