#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "vars.h"
#include "screen.h"
#include "scrsaver.h"
#include "setup.h"
#include "files.h"
#include "keyb.h"

#ifdef __linux__
#  define TABLES_DAT "~/.skyreader/tables.dat"
#else
#  define TABLES_DAT "tables.dat"
#endif

char intab[30], outtab[30];

typedef struct
{
    char name[21];
    char in;
    unsigned char table[256];
}
TTABLE_REC;

static TTABLE_REC ttable;
static char old_setup_area[80] = "";

/* Make default preference settings */
void get_default_settings(void)
{
    memset(&opt,0,sizeof(opt));
    opt.rflags = AUTO_TITLE_SCAN+AUTO_MAIL_SCAN+AUTO_ADVANTAGE+AREAS_ONLY_MSGS+STRIP_RE;
    opt.wflags = INSERT_QUOTE+QUOTE_NOT4U;
    opt.sort_msgs = MSGSORT_SUBJ;
    opt.replysort = REPLYSORT_AREA;
    opt.taglines = TAGLINE_PROMPT;
    opt.quote = QUOTE_GOLDED;
    opt.autosplit = 1000;
#ifdef __linux__
    strcpy(opt.tagfile,"~/.skyreader/default.tag");
    strcpy(opt.adoptfile,"~/.skyreader/default.tag");
    strcpy(opt.signature,"~/.skyreader/skysig.txt");
#else
    strcpy(opt.tagfile,"default.tag");
    strcpy(opt.adoptfile,"default.tag");
    strcpy(opt.signature,"skysig.txt");
#endif
    strcpy(opt.savefile,"@AT@.txt");
    strcpy(opt.replysave,"@AT@.txt");
    strcpy(opt.quotehdr," * Original msg was to @TO@@CR@");
}

void set_defaults(void)
{
    opt.defaults = 0xffffffff;
    opt.rflags = 
        AUTO_TITLE_DEFAULT |
        AUTO_MAIL_DEFAULT |
        NOICE_PERS_DEFAULT |
        AUTO_ADV_DEFAULT |
        AREAS_ONLY_MSGS_DEFAULT |
        TREAT_I_DEFAULT |
        STRIP_RE_DEFAULT;
    opt.wflags =
        INSERT_QUOTE_DEFAULT |
        QUOTE_NOT4U_DEFAULT |
        AUTO_SAVE_DEFAULT |
        REMOVE_SPACES_DEFAULT;
    opt.sort_msgs = MSGSORT_DEF;
    opt.replysort = REPLYSORT_DEF;
    opt.taglines = TAGLINE_DEFAULT;
    opt.quote = QUOTE_DEFAULT;
}

/* Read options */
void load_options(char type, char *area)
{
    char tmp[256],setdefs;
    FILE *Fset;

    setdefs = 0;
    if (type == OPTIONS_DEFAULT)
    {
        /* Read options from default.ini */
        sprintf(tmp,"%sdefault.ini",setup.datapath);
        setdefs = area != NULL;
    }
    else
    {
        /* Read options from <mailpkt>.ini */
        sprintf(tmp,"%s%s.ini",setup.datapath,mailpkt);
    }

    Fset = fopen_ign(tmp,"rb");
    if (Fset == NULL)
    {
        /* Could not open .ini file */
        if (type == OPTIONS_DEFAULT)
        {
            /* Return default settings */
            get_default_settings();
            set_defaults();
            return;
        }
        else
        {
            /* Try to read default.ini */
        __try_it:
            sprintf(tmp,"%sdefault.ini",setup.datapath);
            Fset = fopen_ign(tmp,"rb");
            if (Fset == NULL)
            {
                /* Return default settings */
                get_default_settings();
                set_defaults();
                return;
            }
            setdefs = 1;
            type = OPTIONS_DEFAULT;
        }
    }

    if (type == OPTIONS_AREAOVRRIDE)
    {
        /* Read area preferences */
        while (fread(&opt,sizeof(opt),1,Fset))
        {
            if (strcmp(opt.areatag,area) == 0)
            {
                /* Area found, preferences read.. */
                fclose(Fset);
                return;
            }
        }

        /* Area not found */
        fseek(Fset,0,SEEK_SET);
        setdefs = 1;
    }

    /* Read packet preferences */
    if (fread(&opt,sizeof(opt),1,Fset) == 0)
    {
        /* Could not read them */
        if (type == OPTIONS_DEFAULT)
        {
            /* Return default preferences */
            get_default_settings();
            setdefs = 1;
        }
        else
        {
            /* Try to read from default.ini */
            fclose(Fset);
            goto __try_it;
        }
    }

    if (setdefs) set_defaults();
    fclose(Fset);
}

/* Fill "default" parts of preferences */
void fill_default(void)
{
    if (defopt.rflags & AUTO_TITLE_DEFAULT)
    {
        defopt.rflags &= ~(AUTO_TITLE_SCAN|AUTO_TITLE_DEFAULT);
        defopt.rflags |= opt.rflags & (AUTO_TITLE_SCAN|AUTO_TITLE_DEFAULT);
    }
    if (defopt.rflags & AUTO_MAIL_DEFAULT)
    {
        defopt.rflags &= ~(AUTO_MAIL_SCAN|AUTO_MAIL_DEFAULT);
        defopt.rflags |= opt.rflags & (AUTO_MAIL_SCAN|AUTO_MAIL_DEFAULT);
    }
    if (defopt.rflags & NOICE_PERS_DEFAULT)
    {
        defopt.rflags &= ~(NOICE_PERSONAL|NOICE_PERS_DEFAULT);
        defopt.rflags |= opt.rflags & (NOICE_PERSONAL|NOICE_PERS_DEFAULT);
    }
    if (defopt.rflags & AUTO_ADV_DEFAULT)
    {
        defopt.rflags &= ~(AUTO_ADVANTAGE|AUTO_ADV_DEFAULT);
        defopt.rflags |= opt.rflags & (AUTO_ADVANTAGE|AUTO_ADV_DEFAULT);
    }
    if (defopt.rflags & AREAS_ONLY_MSGS_DEFAULT)
    {
        defopt.rflags &= ~(AREAS_ONLY_MSGS|AREAS_ONLY_MSGS_DEFAULT);
        defopt.rflags |= opt.rflags & (AREAS_ONLY_MSGS|AREAS_ONLY_MSGS_DEFAULT);
    }
    if (defopt.rflags & TREAT_I_DEFAULT)
    {
        defopt.rflags &= ~(TREAT_I_NORMAL|TREAT_I_DEFAULT);
        defopt.rflags |= opt.rflags & (TREAT_I_NORMAL|TREAT_I_DEFAULT);
    }
    if (defopt.rflags & STRIP_RE_DEFAULT)
    {
        defopt.rflags &= ~(STRIP_RE|STRIP_RE_DEFAULT);
        defopt.rflags |= opt.rflags & (STRIP_RE|STRIP_RE_DEFAULT);
    }
    if (defopt.wflags & INSERT_QUOTE_DEFAULT)
    {
        defopt.wflags &= ~(INSERT_QUOTE|INSERT_QUOTE_DEFAULT);
        defopt.wflags |= opt.wflags & (INSERT_QUOTE|INSERT_QUOTE_DEFAULT);
    }
    if (defopt.wflags & REMOVE_SPACES_DEFAULT)
    {
        defopt.wflags &= ~(REMOVE_SPACES|REMOVE_SPACES_DEFAULT);
        defopt.wflags |= opt.wflags & (REMOVE_SPACES|REMOVE_SPACES_DEFAULT);
    }
    if (defopt.wflags & QUOTE_NOT4U_DEFAULT)
    {
        defopt.wflags &= ~(QUOTE_NOT4U|QUOTE_NOT4U_DEFAULT);
        defopt.wflags |= opt.wflags & (QUOTE_NOT4U|QUOTE_NOT4U_DEFAULT);
    }
    if (defopt.wflags & AUTO_SAVE_DEFAULT)
    {
        defopt.wflags &= ~(AUTO_SAVE|AUTO_SAVE_DEFAULT);
        defopt.wflags |= opt.wflags & (AUTO_SAVE|AUTO_SAVE_DEFAULT);
    }
    if (defopt.wflags2 & SUBJ_HEADER_DEFAULT)
    {
        defopt.wflags2 &= ~(SUBJ_HEADER|SUBJ_HEADER_DEFAULT);
        defopt.wflags2 |= opt.wflags2 & (SUBJ_HEADER|SUBJ_HEADER_DEFAULT);
    }

    if (defopt.quote == QUOTE_DEFAULT) defopt.quote = opt.quote;
    if (defopt.taglines == TAGLINE_DEFAULT) defopt.taglines = opt.taglines;
    if (defopt.sort_msgs == MSGSORT_DEF) defopt.sort_msgs = opt.sort_msgs;
    if (defopt.replysort == REPLYSORT_DEF) defopt.replysort = opt.replysort;

    if (defopt.defaults & DEFAULT_AUTOSPLIT)
    {
        defopt.autosplit = opt.autosplit;
        if ((opt.defaults & DEFAULT_AUTOSPLIT) == 0) defopt.defaults ^= DEFAULT_AUTOSPLIT;
    }
    if (defopt.defaults & DEFAULT_INTABLE)
    {
        strcpy(defopt.intable,opt.intable);
        if ((opt.defaults & DEFAULT_INTABLE) == 0) defopt.defaults ^= DEFAULT_INTABLE;
    }
    if (defopt.defaults & DEFAULT_OUTTABLE)
    {
        strcpy(defopt.outtable,opt.outtable);
        if ((opt.defaults & DEFAULT_OUTTABLE) == 0) defopt.defaults ^= DEFAULT_OUTTABLE;
    }
    if (defopt.defaults & DEFAULT_TAGFILE)
    {
        strcpy(defopt.tagfile,opt.tagfile);
        if ((opt.defaults & DEFAULT_TAGFILE) == 0) defopt.defaults ^= DEFAULT_TAGFILE;
    }
    if (defopt.defaults & DEFAULT_ADOPTFILE)
    {
        strcpy(defopt.adoptfile,opt.adoptfile);
        if ((opt.defaults & DEFAULT_ADOPTFILE) == 0) defopt.defaults ^= DEFAULT_ADOPTFILE;
    }
    if (defopt.defaults & DEFAULT_SIGNATURE)
    {
        strcpy(defopt.signature,opt.signature);
        if ((opt.defaults & DEFAULT_SIGNATURE) == 0) defopt.defaults ^= DEFAULT_SIGNATURE;
    }
    if (defopt.defaults & DEFAULT_SAVEFILE)
    {
        strcpy(defopt.savefile,opt.savefile);
        if ((opt.defaults & DEFAULT_SAVEFILE) == 0) defopt.defaults ^= DEFAULT_SAVEFILE;
    }
    if (defopt.defaults & DEFAULT_REPLYSAVE)
    {
        strcpy(defopt.replysave,opt.replysave);
        if ((opt.defaults & DEFAULT_REPLYSAVE) == 0) defopt.defaults ^= DEFAULT_REPLYSAVE;
    }
    if (defopt.defaults & DEFAULT_KEYWORD)
    {
        strcpy(defopt.keyword,opt.keyword);
        if ((opt.defaults & DEFAULT_KEYWORD) == 0) defopt.defaults ^= DEFAULT_KEYWORD;
    }
    if (defopt.defaults & DEFAULT_ALIAS)
    {
        strcpy(defopt.alias,opt.alias);
        if ((opt.defaults & DEFAULT_ALIAS) == 0) defopt.defaults ^= DEFAULT_ALIAS;
    }
    if ((defopt.defaults & DEFAULT_EDITOR) || (defopt.editor[0] == 0))
    {
        strcpy(defopt.editor,opt.editor);
        if ((opt.defaults & DEFAULT_EDITOR) == 0) defopt.defaults &= ~DEFAULT_EDITOR;
    }
    if (defopt.defaults & DEFAULT_QUOTEHDR)
    {
        strcpy(defopt.quotehdr,opt.quotehdr);
        if ((opt.defaults & DEFAULT_QUOTEHDR) == 0) defopt.defaults ^= DEFAULT_QUOTEHDR;
    }
}

/* Read area setup */
void read_setup(char *area)
{
    if (area == NULL)
    {
        old_setup_area[0] = '\0';

        /* Read packet preferences */
        strcpy(mailpkt,pktname);
        load_options(OPTIONS_PKTOVRRIDE,NULL);
        memcpy(&defopt,&opt,sizeof(opt));

        /* Read default preferences */
        load_options(OPTIONS_DEFAULT,NULL);
        fill_default();

        get_default_settings();
        fill_default();

        memcpy(&opt,&defopt,sizeof(opt));
    }
    else
    {
        /* Read area preferences */
        if (strcmp(area,old_setup_area) == 0) return;
        strcpy(old_setup_area,area);

        /* Read area preferences */
        strcpy(mailpkt,pktname);
        load_options(OPTIONS_AREAOVRRIDE,area);
        memcpy(&defopt,&opt,sizeof(opt));

        /* Read packet preferences */
        load_options(OPTIONS_PKTOVRRIDE,NULL);
        fill_default();

        /* Read default.ini preferences */
        load_options(OPTIONS_DEFAULT,NULL);
        fill_default();

        /* Get default preferences */
        get_default_settings();
        fill_default();

        memcpy(&opt,&defopt,sizeof(opt));
    }
}

/* Read translation table */
void read_table(char in, unsigned char *table, char *tablename)
{
    FILE *Ftab;
    int num;

    if (table == inbound)
        strcpy(intab, tablename);
    else if (table == outbound)
        strcpy(outtab, tablename);

    /* Open tables.dat file */
    Ftab = FileOpen(TABLES_DAT,"rb");
    if (Ftab == NULL) return;

    while (fread(&ttable,sizeof(ttable),1,Ftab))
    {
        if ((in == ttable.in) && (strcmp(tablename,ttable.name) == 0))
        {
            /* Table found */
            memcpy(table,ttable.table,sizeof(ttable.table));
            fclose(Ftab);
            return;
        }
    }
    fclose(Ftab);

    if (table == inbound)
        intab[0] = '\0';
    else if (table == outbound)
        outtab[0] = '\0';

    /* Not found, use default table */
    for (num=0; num<=255; num++) table[num] = (unsigned char) num;
}

/* Read translation table, if it's not found, don't fill with default */
void read_table2(char in, unsigned char *table, char *tablename)
{
    FILE *Ftab;

    /* Open tables.dat file */
    Ftab = FileOpen(TABLES_DAT,"rb");
    if (Ftab == NULL) return;

    while (fread(&ttable,sizeof(ttable),1,Ftab))
    {
        if ((in == ttable.in) && (strcmp(tablename,ttable.name) == 0))
        {
            /* Table found */
            if (table == inbound)
                strcpy(intab, tablename);
            else if (table == outbound)
                strcpy(outtab, tablename);

            memcpy(table,ttable.table,sizeof(ttable.table));
            fclose(Ftab);
            return;
        }
    }
    fclose(Ftab);
}

static void draw_line_tt(char ypos, char pos)
{
    char str3[4];
    
    tattrbar(scrwidth/2-22,ypos,scrwidth/2+21,ypos,(1 << 4)+15);
    sprintf(str3,"%3i",pos);
    cwritexy(scrwidth/2-21,ypos,str3,(1 << 4)+15);
    if (ttable.in)
        cwritexy(scrwidth/2-17,ypos,"In ",(1 << 4)+15);
    else
        cwritexy(scrwidth/2-17,ypos,"Out",(1 << 4)+15);
    cwritexy(scrwidth/2-13,ypos,ttable.name,(1 << 4)+15);
}

char ask_delete(void)
{
    unsigned short cx,cy;
    char *oldscr;

    char quit,yes,ch;

    save_scr(&cx,&cy,&oldscr);
    draw_shaded_box(scrwidth/2-20,9,scrwidth/2+21,14,(4 << 4)+12,(4 << 4)+15,"Really delete translation table?");
    cmiddle(11,"Yes, please",4 << 4);
    cmiddle(12,"No, don't!",4 << 4);
    tattrbar(scrwidth/2-19,12,scrwidth/2+20,12,7 << 4);
    yes = 0; quit = 0;
    while (!quit) {
        if (sk_kbhit()) {
            ch = toupper(sk_getch());
            switch (ch) {
                case 13:
                    quit = 1;
                    break;
                case 'Y':
                    yes = 1; goto drawyes;
                case 'N':
                    yes = 0; goto drawno;
                case 0:
                    ch = sk_getch();
                    if ((ch == 'H') || (ch == 'P')) {
                        yes = !yes;
                        if (!yes) {
                        drawno:
                            tattrbar(scrwidth/2-19,11,scrwidth/2+20,11,4 << 4);
                            tattrbar(scrwidth/2-19,12,scrwidth/2+20,12,7 << 4);
                        } else {
                        drawyes:
                            tattrbar(scrwidth/2-19,11,scrwidth/2+20,11,7 << 4);
                            tattrbar(scrwidth/2-19,12,scrwidth/2+20,12,4 << 4);
                        }
                    }
            }
        } else give_timeslice();
    }
           
    old_scr(cx,cy,&oldscr);
    return yes;
}

void edit_table(void)
{
    char ox,oy,x,y,col,ch,tmp[30];
    char info,add,quit;

    strcpy(tmp,ttable.name);
    if (ttable.in) strcat(tmp," (In)"); else strcat(tmp," (Out)");
    draw_shaded_box(scrwidth/2-35,3,scrwidth/2+36,5,(1 << 4)+9,(1 << 4)+15,tmp);
    cmiddle(4,"ALT-N = Table name, ALT-T = Table type (in/out), ENTER = edit",(1 << 4)+7);
    draw_shaded_box(scrwidth/2-39,7,scrwidth/2+40,24,(1 << 4)+9,(1 << 4)+15,NULL);

    for (y=0; y<=15; y++)
    {
        for (x=0; x<=15; x++)
        {
            writechr(scrwidth/2-40+x*5+2,y+8,scrwidth/2-40+y+x*16,(1 << 4)+15);
            writechr(scrwidth/2-40+x*5+3,y+8,scrwidth/2-40+ttable.table[y+x*16],(1 << 4)+7);
        }
    }
    x = 0; y = 0; ox = 0; oy = 0;
    add = 3; col = (1 << 4)+7;
    quit = 0;
    tattr(scrwidth/2-40+x*5+add,y+8,4 << 4);
    while (!quit)
    {
        if (sk_kbhit())
        {
            ch = sk_getch();
            switch (ch)
            {
                case 27:
                    quit = 1;
                    break;
                case 13:
                    if (add == 3)
                    {
                        ox = x; oy = y;
                        col = (1 << 4)+15;
                        add = 2;
                        tattr(scrwidth/2-40+x*5+add,y+8,4 << 4);
                    }
                    else
                    {
                        tattr(scrwidth/2-40+x*5+add,y+8,col);
                        ttable.table[oy+ox*16] = y+x*16;
                        writechr(scrwidth/2-40+ox*5+3,oy+8,ttable.table[oy+ox*16],(1 << 4)+7);
                        col = (1 << 4)+7;
                        add = 3;
                        x = ox; y = oy;
                        tattr(scrwidth/2-40+ox*5+add,oy+8,col);
                        tattr(scrwidth/2-40+x*5+add,y+8,4 << 4);
                    }
                    break;
                case 0:
                    ch = sk_getch();
                    switch (ch)
                    {
                        case 49:
                            tbar(scrwidth/2-34,4,scrwidth/2+35,4,(1 << 4)+15);
                            cwritexy(scrwidth/2-33,4,"Charset:",(1 << 4)+15);
                            info = get_string(scrwidth/2-24,4,20,20,ttable.name,1,(1 << 4)+11,
                                              (1 << 4)+15,(1 << 4)+10);
                            tbar(scrwidth/2-34,4,scrwidth/2+35,4,(1 << 4)+15);
                            strcpy(tmp,ttable.name);
                            if (ttable.in) strcat(tmp," (In)"); else strcat(tmp," (Out)");
                            draw_shaded_box(scrwidth/2-35,3,scrwidth/2+36,5,(1 << 4)+9,(1 << 4)+15,tmp);
                            cmiddle(4,"ALT-N = Table name, ALT-T = Table type (in/out), ENTER = edit",(1 << 4)+7);
                            break;
                        case 20:
                            ttable.in = !ttable.in;
                            strcpy(tmp,ttable.name);
                            if (ttable.in) strcat(tmp," (In)"); else strcat(tmp," (Out)");
                            draw_shaded_box(scrwidth/2-35,3,scrwidth/2+36,5,(1 << 4)+9,(1 << 4)+15,tmp);
                            cmiddle(4,"ALT-N = Table name, ALT-T = Table type (in/out), ENTER = edit",(1 << 4)+7);
                            break;
                        case 'H':
                            tattr(scrwidth/2-40+x*5+add,y+8,col);
                            if (y == 0) y = 15; else y--;
                            tattr(scrwidth/2-40+x*5+add,y+8,4 << 4);
                            break;
                        case 'P':
                            tattr(scrwidth/2-40+x*5+add,y+8,col);
                            if (y == 15) y = 0; else y++;
                            tattr(scrwidth/2-40+x*5+add,y+8,4 << 4);
                            break;
                        case 'K':
                            tattr(scrwidth/2-40+x*5+add,y+8,col);
                            if (x == 0) x = 15; else x--;
                            tattr(scrwidth/2-40+x*5+add,y+8,4 << 4);
                            break;
                        case 'M':
                            tattr(scrwidth/2-40+x*5+add,y+8,col);
                            if (x == 15) x = 0; else x++;
                            tattr(scrwidth/2-40+x*5+add,y+8,4 << 4);
                            break;
                    }
                    break;
                default:
                    tattr(scrwidth/2-40+x*5+add,y+8,col);
                    y = ch & 15;
                    x = ch >> 4;
                    tattr(scrwidth/2-40+x*5+add,y+8,4 << 4);
            }
        } else give_timeslice();
    }
}

char *transtable(char *str, unsigned char *table)
{
    unsigned short cx,cy;
    char *oldscr;

    FILE *Ftab;

    int max,pos,tables;
    char ypos,upy,ch;
    unsigned num;
    char quit;

    if ((Ftab = FileOpen(TABLES_DAT,"r+b")) == NULL)
        if ((Ftab = FileOpen(TABLES_DAT,"w+b")) == NULL) return str;
    fseek(Ftab,0,SEEK_END);
    tables = ftell(Ftab) / sizeof(ttable);

    ypos = 1; upy = 0;
__again:
    save_scr(&cx,&cy,&oldscr);
    draw_shaded_box(scrwidth/2-35,3,scrwidth/2+36,5,(1 << 4)+9,(1 << 4)+15,"Translation table setup");
    cmiddle(4,"(INS)ert/(DEL)ete/(E)dit, (ENTER) to select or use (D)efault table",(1 << 4)+7);
    draw_shaded_box(scrwidth/2-23,7,scrwidth/2+22,23,(1 << 4)+9,(1 << 4)+15,NULL);
    max = tables-upy;
    if (max > 15) max = 15;

    fseek(Ftab,upy*sizeof(ttable),SEEK_SET);
    for (pos=1; pos<=max; pos++)
    {
        fread(&ttable,sizeof(ttable),1,Ftab);
        draw_line_tt(7+pos,pos+upy);
    }
               
    tattrbar(scrwidth/2-22,ypos+7,scrwidth/2+21,ypos+7,7 << 4);
    quit = 0;
    while (!quit)
    {
        if (sk_kbhit())
        {
            ch = toupper(sk_getch());
            switch (ch)
            {
                case 27:
                    quit = 1;
                    break;
                case 13:
                    fseek(Ftab,(ypos+upy-1)*sizeof(ttable),SEEK_SET);
                    fread(&ttable,sizeof(ttable),1,Ftab);
                    if (str != NULL) strcpy(str,ttable.name);
                    if (table != NULL)
                    {
                        if (table == inbound)
                            strcpy(intab, ttable.name);
                        else if (table == outbound)
                            strcpy(outtab, ttable.name);
                        memcpy(table,ttable.table,256);
                    }
                    quit = 1;
                    break;
                case 'D':
                    if (table != NULL)
                    {
                        if (table == inbound)
                            intab[0] = '\0';
                        else if (table == outbound)
                            outtab[0] = '\0';
                        for (num=0; num<=255; num++)
                            table[num] = (unsigned char) num;
                    }
                    if (str != NULL) str[0] = 0;
                    quit = 1;
                    break;
                case 'E':
                    if (ypos+upy <= tables)
                    {
                        old_scr(cx,cy,&oldscr);
                        fseek(Ftab,(ypos+upy-1)*sizeof(ttable),SEEK_SET);
                        fread(&ttable,sizeof(ttable),1,Ftab);
                        save_scr(&cx,&cy,&oldscr);
                        edit_table();
                        old_scr(cx,cy,&oldscr);
                        fseek(Ftab,(ypos+upy-1)*sizeof(ttable),SEEK_SET);
                        fwrite(&ttable,sizeof(ttable),1,Ftab);
                        goto __again;
                    }
                    break;
                case 0:
                    ch = sk_getch();
                    switch (ch)
                    {
                        case 'S':
                            if (ypos+upy <= tables)
                            {
                                if (!ask_delete()) continue;
                                tbar(scrwidth/2-22,8,scrwidth/2+21,22,(1 << 4)+15);
                                delete_space(Ftab,(ypos+upy-1)*sizeof(ttable),sizeof(ttable),tables*sizeof(ttable));
                                tables--;
                                ftrunc(Ftab,tables*sizeof(ttable));
                                fseek(Ftab,0,SEEK_SET);
                                max = tables-upy;
                                if (max > 15) max = 15;
                                for (pos=1; pos<=max; pos++)
                                {
                                    fread(&ttable,sizeof(ttable),1,Ftab);
                                    draw_line_tt(7+pos,pos+upy);
                                }
                                if ((ypos+upy > tables) && (tables > 0))
                                {
                                    tbar(scrwidth/2-22,ypos+7,scrwidth/2+21,ypos+7,(1 << 4)+15);
                                    goto __goup;
                                }
                                tattrbar(scrwidth/2-22,ypos+7,scrwidth/2+21,ypos+7,7 << 4);
                            }
                            break;
                        case 'R':
                            if (tables < 255)
                            {
                                fseek(Ftab,tables*sizeof(ttable),SEEK_SET);
                                strcpy(ttable.name,"UNKNOWN");
                                for (pos=0; pos<=255; pos++)
                                    ttable.table[pos] = pos;
                                fwrite(&ttable,sizeof(ttable),1,Ftab);
                                tables++;
                                if (tables-upy <= 15)
                                {
                                    draw_line_tt(tables-upy+7,tables);
                                    tattrbar(scrwidth/2-22,ypos+7,scrwidth/2+21,ypos+7,7 << 4);
                                }
                            }
                            break;
                        case 'H':
                            if (ypos+upy > 1)
                            {
                                fseek(Ftab,(ypos+upy-1)*sizeof(ttable),SEEK_SET);
                                fread(&ttable,sizeof(ttable),1,Ftab);
                                draw_line_tt(7+ypos,ypos+upy);
                            __goup:
                                if (ypos == 1)
                                {
                                    scroll_down(scrwidth/2-22,8,scrwidth/2+21,22);
                                    upy--;
                                    draw_line_tt(7+ypos,ypos+upy);
                                } else ypos--;
                                tattrbar(scrwidth/2-22,ypos+7,scrwidth/2+21,ypos+7,7 << 4);
                            }
                            break;
                        case 'P':
                            if (ypos+upy < tables)
                            {
                                fseek(Ftab,(ypos+upy-1)*sizeof(ttable),SEEK_SET);
                                fread(&ttable,sizeof(ttable),1,Ftab);
                                draw_line_tt(7+ypos,ypos+upy);
                                if (ypos == 15)
                                {
                                    scroll_up(scrwidth/2-22,8,scrwidth/2+21,22);
                                    upy++;
                                    draw_line_tt(7+ypos,ypos+upy);
                                }
                                else
                                    ypos++;
                                tattrbar(scrwidth/2-22,ypos+7,scrwidth/2+21,ypos+7,7 << 4);
                            }
                            break;
                    }
                    break;
            }
        } else give_timeslice();
    }
    old_scr(cx,cy,&oldscr);
    fclose(Ftab);
    return str;
}
